<?php
/**
 * sites.php - API de gestion des sites et affectations utilisateurs
 *
 * Endpoints:
 * - GET /sites.php - Liste tous les sites disponibles
 * - GET /sites.php?action=user_sites&user_id=X - Sites d'un utilisateur spécifique
 * - POST /sites.php - Assigner des sites à un utilisateur
 * - DELETE /sites.php - Retirer tous les sites d'un utilisateur
 * - GET /sites.php?action=stats - Statistiques sur l'utilisation des sites
 */

require_once 'config.php';
require_once 'JwtManager.php';
require_once 'SiteAccessControl.php';

// Headers CORS
Utils::setCorsHeaders();

$method = $_SERVER['REQUEST_METHOD'];

try {
    // Vérifier l'authentification pour toutes les requêtes
    $payload = JwtManager::requireAuth();

    switch ($method) {
        case 'GET':
            $action = $_GET['action'] ?? 'list';

            switch ($action) {
                case 'list':
                    // Lister tous les sites disponibles
                    // Accessible par tous les utilisateurs authentifiés
                    $sites = SiteAccessControl::getAllAvailableSites();

                    Utils::jsonResponse([
                        'sites' => $sites,
                        'count' => count($sites)
                    ], true, count($sites) . ' sites disponibles');
                    break;

                case 'user_sites':
                    // Récupérer les sites d'un utilisateur spécifique
                    $userId = $_GET['user_id'] ?? null;

                    if (!$userId) {
                        Utils::errorResponse('ID utilisateur requis');
                    }

                    // Vérifier les permissions
                    // Seuls les admins ou l'utilisateur lui-même peuvent voir ses sites
                    if ($payload['groupe'] !== 'ADMINISTRATEUR' && $payload['user_id'] != $userId) {
                        Utils::errorResponse('Accès refusé', 403);
                    }

                    $userSites = SiteAccessControl::getUserSites($userId);

                    Utils::jsonResponse([
                        'user_id' => (int)$userId,
                        'sites' => $userSites,
                        'count' => count($userSites),
                        'has_restriction' => !empty($userSites)
                    ], true, 'Sites de l\'utilisateur récupérés');
                    break;

                case 'stats':
                    // Statistiques (admin uniquement)
                    if ($payload['groupe'] !== 'ADMINISTRATEUR') {
                        Utils::errorResponse('Accès refusé - Administrateur requis', 403);
                    }

                    $stats = SiteAccessControl::getSiteStatistics();

                    Utils::jsonResponse($stats, true, 'Statistiques récupérées');
                    break;

                case 'my_sites':
                    // Récupérer les sites de l'utilisateur connecté
                    $userSites = SiteAccessControl::getUserSites($payload['user_id']);

                    Utils::jsonResponse([
                        'sites' => $userSites,
                        'count' => count($userSites),
                        'has_restriction' => !empty($userSites)
                    ], true, 'Vos sites');
                    break;

                default:
                    Utils::errorResponse('Action non reconnue');
            }
            break;

        case 'POST':
            // Assigner des sites à un utilisateur (admin uniquement)
            if ($payload['groupe'] !== 'ADMINISTRATEUR') {
                Utils::errorResponse('Accès refusé - Administrateur requis', 403);
            }

            $input = Utils::getJsonInput();

            // Validation des entrées
            if (!isset($input['user_id'])) {
                Utils::errorResponse('ID utilisateur requis');
            }

            if (!isset($input['sites']) || !is_array($input['sites'])) {
                Utils::errorResponse('Liste des sites requise (tableau)');
            }

            $userId = (int)$input['user_id'];
            $sites = $input['sites'];

            // Valider le format de chaque site
            foreach ($sites as $site) {
                if (!isset($site['code']) || !isset($site['nom'])) {
                    Utils::errorResponse('Format de site invalide. Chaque site doit avoir "code" et "nom"');
                }

                // Valider le rôle si fourni
                if (isset($site['role']) && !in_array($site['role'], ['lecteur', 'editeur', 'admin'])) {
                    Utils::errorResponse('Rôle invalide. Valeurs acceptées: lecteur, editeur, admin');
                }
            }

            // Assigner les sites
            try {
                $result = SiteAccessControl::assignSitesToUser($userId, $sites);

                if ($result) {
                    Utils::jsonResponse([
                        'user_id' => $userId,
                        'sites_assigned' => count($sites)
                    ], true, 'Sites assignés avec succès');
                } else {
                    Utils::errorResponse('Échec de l\'assignation des sites');
                }
            } catch (Exception $e) {
                Utils::errorResponse('Erreur lors de l\'assignation: ' . $e->getMessage());
            }
            break;

        case 'DELETE':
            // Retirer tous les sites d'un utilisateur (admin uniquement)
            if ($payload['groupe'] !== 'ADMINISTRATEUR') {
                Utils::errorResponse('Accès refusé - Administrateur requis', 403);
            }

            $input = Utils::getJsonInput();

            if (!isset($input['user_id'])) {
                Utils::errorResponse('ID utilisateur requis');
            }

            $userId = (int)$input['user_id'];

            $result = SiteAccessControl::removeAllSites($userId);

            if ($result) {
                Utils::jsonResponse([
                    'user_id' => $userId
                ], true, 'Tous les sites ont été retirés. L\'utilisateur a maintenant un accès illimité.');
            } else {
                Utils::errorResponse('Échec de la suppression des sites');
            }
            break;

        case 'PUT':
            // Mettre à jour les sites d'un utilisateur (admin uniquement)
            if ($payload['groupe'] !== 'ADMINISTRATEUR') {
                Utils::errorResponse('Accès refusé - Administrateur requis', 403);
            }

            $input = Utils::getJsonInput();

            if (!isset($input['user_id']) || !isset($input['sites'])) {
                Utils::errorResponse('ID utilisateur et sites requis');
            }

            $userId = (int)$input['user_id'];
            $sites = $input['sites'];

            // Valider le format
            if (!is_array($sites)) {
                Utils::errorResponse('Le champ sites doit être un tableau');
            }

            try {
                $result = SiteAccessControl::assignSitesToUser($userId, $sites);

                if ($result) {
                    Utils::jsonResponse([
                        'user_id' => $userId,
                        'sites_count' => count($sites)
                    ], true, 'Sites mis à jour avec succès');
                } else {
                    Utils::errorResponse('Échec de la mise à jour des sites');
                }
            } catch (Exception $e) {
                Utils::errorResponse('Erreur lors de la mise à jour: ' . $e->getMessage());
            }
            break;

        default:
            Utils::errorResponse('Méthode HTTP non supportée', 405);
    }

} catch (Exception $e) {
    Utils::logError("Erreur dans sites.php", ['error' => $e->getMessage()]);
    Utils::errorResponse('Erreur interne du serveur: ' . $e->getMessage(), 500);
}
?>
