/**
 * site-filter-client.js
 * Gestion côté client du filtrage par sites
 *
 * Ce module gère:
 * - Le stockage des sites accessibles de l'utilisateur
 * - Le filtrage automatique des requêtes API
 * - L'affichage d'un sélecteur de site si l'utilisateur a plusieurs sites
 * - La mise à jour des données en fonction du site sélectionné
 */

const SiteFilterClient = (function() {
    'use strict';

    // Variables privées
    let userSites = [];
    let currentSite = null;
    let hasSiteRestriction = false;

    /**
     * Initialiser le module avec les sites de l'utilisateur
     * @param {Object} userData - Données utilisateur du login
     */
    function init(userData) {
        if (!userData) {
            console.error('SiteFilterClient: userData requis pour l\'initialisation');
            return;
        }

        // Récupérer les sites de l'utilisateur
        userSites = userData.sites || [];
        hasSiteRestriction = userData.has_site_restriction || false;

        // Stocker dans localStorage
        localStorage.setItem('userSites', JSON.stringify(userSites));
        localStorage.setItem('hasSiteRestriction', hasSiteRestriction);

        // Si l'utilisateur a des sites, sélectionner le premier par défaut
        if (userSites.length > 0) {
            currentSite = userSites[0];
            localStorage.setItem('currentSite', JSON.stringify(currentSite));
        }

        console.log('SiteFilterClient initialisé:', {
            sites: userSites.length,
            hasRestriction: hasSiteRestriction,
            currentSite: currentSite
        });

        // Afficher le sélecteur de site si plusieurs sites
        if (userSites.length > 1) {
            displaySiteSelector();
        } else if (userSites.length === 1) {
            displayCurrentSite();
        }
    }

    /**
     * Charger les données depuis localStorage
     */
    function loadFromStorage() {
        try {
            const storedSites = localStorage.getItem('userSites');
            const storedCurrentSite = localStorage.getItem('currentSite');
            const storedRestriction = localStorage.getItem('hasSiteRestriction');

            if (storedSites) {
                userSites = JSON.parse(storedSites);
            }

            if (storedCurrentSite) {
                currentSite = JSON.parse(storedCurrentSite);
            }

            if (storedRestriction) {
                hasSiteRestriction = storedRestriction === 'true';
            }
        } catch (error) {
            console.error('Erreur lors du chargement des sites depuis localStorage:', error);
        }
    }

    /**
     * Afficher le sélecteur de site (pour utilisateurs avec plusieurs sites)
     */
    function displaySiteSelector() {
        const container = document.getElementById('siteSelectorContainer');

        if (!container) {
            // Créer le conteneur s'il n'existe pas
            const newContainer = document.createElement('div');
            newContainer.id = 'siteSelectorContainer';
            newContainer.style.cssText = `
                position: fixed;
                top: 70px;
                right: 20px;
                z-index: 1000;
                background: white;
                padding: 15px;
                border-radius: 10px;
                box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            `;
            document.body.appendChild(newContainer);
        }

        const html = `
            <div style="margin-bottom: 5px; font-weight: 600; color: #333;">
                🌐 Site actif:
            </div>
            <select id="siteSelector" onchange="SiteFilterClient.changeSite(this.value)"
                    style="width: 200px; padding: 8px; border: 1px solid #ddd; border-radius: 5px;">
                ${userSites.map(site => `
                    <option value="${site.code}" ${currentSite && currentSite.code === site.code ? 'selected' : ''}>
                        ${site.nom}
                    </option>
                `).join('')}
            </select>
        `;

        const target = document.getElementById('siteSelectorContainer');
        if (target) {
            target.innerHTML = html;
        }
    }

    /**
     * Afficher le site unique (pour utilisateurs avec un seul site)
     */
    function displayCurrentSite() {
        const container = document.getElementById('siteSelectorContainer');

        if (!container) {
            const newContainer = document.createElement('div');
            newContainer.id = 'siteSelectorContainer';
            newContainer.style.cssText = `
                position: fixed;
                top: 70px;
                right: 20px;
                z-index: 1000;
                background: white;
                padding: 10px 15px;
                border-radius: 10px;
                box-shadow: 0 4px 6px rgba(0,0,0,0.1);
            `;
            document.body.appendChild(newContainer);
        }

        const html = `
            <div style="color: #667eea; font-weight: 600;">
                🌐 ${currentSite ? currentSite.nom : 'Tous les sites'}
            </div>
        `;

        const target = document.getElementById('siteSelectorContainer');
        if (target) {
            target.innerHTML = html;
        }
    }

    /**
     * Changer le site actif
     * @param {string} siteCode - Code du nouveau site
     */
    function changeSite(siteCode) {
        const site = userSites.find(s => s.code === siteCode);

        if (site) {
            currentSite = site;
            localStorage.setItem('currentSite', JSON.stringify(currentSite));

            console.log('Site changé:', currentSite);

            // Déclencher un événement personnalisé
            const event = new CustomEvent('siteChanged', {
                detail: { site: currentSite }
            });
            window.dispatchEvent(event);

            // Recharger les données si une fonction de callback existe
            if (window.onSiteChanged && typeof window.onSiteChanged === 'function') {
                window.onSiteChanged(currentSite);
            }
        }
    }

    /**
     * Obtenir le site actuel
     * @returns {Object|null} Site actuel
     */
    function getCurrentSite() {
        return currentSite;
    }

    /**
     * Obtenir tous les sites de l'utilisateur
     * @returns {Array} Liste des sites
     */
    function getUserSites() {
        return userSites;
    }

    /**
     * Vérifier si l'utilisateur a une restriction de sites
     * @returns {boolean}
     */
    function hasRestriction() {
        return hasSiteRestriction;
    }

    /**
     * Ajouter les paramètres de site à une URL
     * @param {string} url - URL de base
     * @returns {string} URL avec paramètres de site
     */
    function addSiteParamsToUrl(url) {
        if (!hasSiteRestriction || !currentSite) {
            return url;
        }

        const separator = url.includes('?') ? '&' : '?';
        return `${url}${separator}codeSite=${encodeURIComponent(currentSite.code)}`;
    }

    /**
     * Wrapper pour fetch avec ajout automatique du filtre de site
     * @param {string} url - URL à appeler
     * @param {Object} options - Options fetch
     * @returns {Promise} Promise du fetch
     */
    async function fetchWithSiteFilter(url, options = {}) {
        // Ajouter le token d'authentification
        const token = localStorage.getItem('token');

        if (!options.headers) {
            options.headers = {};
        }

        if (token) {
            options.headers['Authorization'] = 'Bearer ' + token;
        }

        options.headers['Content-Type'] = 'application/json';

        // Pour les requêtes GET, ajouter le paramètre de site dans l'URL
        if (!options.method || options.method === 'GET') {
            url = addSiteParamsToUrl(url);
        }

        // Pour les requêtes POST/PUT, ajouter le site dans le body
        if ((options.method === 'POST' || options.method === 'PUT') && currentSite) {
            const body = options.body ? JSON.parse(options.body) : {};
            body.site_filter = currentSite.code;
            options.body = JSON.stringify(body);
        }

        return fetch(url, options);
    }

    /**
     * Créer un badge visuel du site actuel
     * @param {string} containerId - ID du conteneur
     */
    function createSiteBadge(containerId) {
        const container = document.getElementById(containerId);

        if (!container) {
            console.warn('Conteneur non trouvé:', containerId);
            return;
        }

        if (!hasSiteRestriction) {
            container.innerHTML = '<span style="color: #28a745;">✓ Tous les sites</span>';
            return;
        }

        if (currentSite) {
            container.innerHTML = `
                <span style="display: inline-block; padding: 5px 10px; background: #667eea;
                             color: white; border-radius: 5px; font-size: 12px;">
                    🌐 ${currentSite.nom}
                </span>
            `;
        } else {
            container.innerHTML = '<span style="color: #dc3545;">⚠ Aucun site sélectionné</span>';
        }
    }

    /**
     * Filtrer un tableau de pesées par site actuel (filtrage client-side)
     * @param {Array} pesees - Tableau de pesées
     * @returns {Array} Pesées filtrées
     */
    function filterPeseesBySite(pesees) {
        if (!hasSiteRestriction || !currentSite) {
            return pesees;
        }

        return pesees.filter(pesee => pesee.CodeSite === currentSite.code);
    }

    /**
     * Obtenir les codes de sites pour les requêtes SQL/API
     * @returns {string|null} Codes de sites séparés par virgule
     */
    function getSiteCodesForQuery() {
        if (!hasSiteRestriction || userSites.length === 0) {
            return null;
        }

        return userSites.map(s => s.code).join(',');
    }

    /**
     * Vérifier si l'utilisateur a accès à un site spécifique
     * @param {string} siteCode - Code du site
     * @returns {boolean}
     */
    function canAccessSite(siteCode) {
        if (!hasSiteRestriction) {
            return true;
        }

        return userSites.some(s => s.code === siteCode);
    }

    /**
     * Nettoyer les données (à appeler lors de la déconnexion)
     */
    function cleanup() {
        userSites = [];
        currentSite = null;
        hasSiteRestriction = false;

        localStorage.removeItem('userSites');
        localStorage.removeItem('currentSite');
        localStorage.removeItem('hasSiteRestriction');

        const container = document.getElementById('siteSelectorContainer');
        if (container) {
            container.remove();
        }
    }

    /**
     * Afficher une alerte si l'utilisateur essaie d'accéder à un site non autorisé
     * @param {string} siteCode - Code du site
     */
    function showAccessDenied(siteCode) {
        alert(`⛔ Accès refusé au site: ${siteCode}\n\nVous n'êtes pas autorisé à accéder à ce site.`);
    }

    // API publique
    return {
        init,
        loadFromStorage,
        changeSite,
        getCurrentSite,
        getUserSites,
        hasRestriction,
        addSiteParamsToUrl,
        fetchWithSiteFilter,
        createSiteBadge,
        filterPeseesBySite,
        getSiteCodesForQuery,
        canAccessSite,
        cleanup,
        showAccessDenied
    };
})();

// Écouter les changements de site
window.addEventListener('siteChanged', function(event) {
    console.log('Événement siteChanged détecté:', event.detail);
});

// Auto-initialisation depuis localStorage si disponible
if (typeof window !== 'undefined') {
    document.addEventListener('DOMContentLoaded', function() {
        SiteFilterClient.loadFromStorage();
    });
}
