// pdf-export-separated.js - Export PDF en deux fichiers séparés

/**
 * Export PDF séparé - Graphiques en portrait
 */
async function exportGraphiquesToPDF() {
    try {
        showTemporaryMessage('📊 Génération PDF Graphiques...', 'warning');
        
        const { jsPDF } = window.jspdf;
        const pdf = new jsPDF('portrait', 'mm', 'a4');
        
        // Configuration UTF-8
        pdf.setFont("helvetica");
        pdf.setCharSpace(0);
        
        const pageWidth = pdf.internal.pageSize.getWidth();
        const pageHeight = pdf.internal.pageSize.getHeight();
        const margin = 15;
        let currentY = margin;
        
        // === EN-TÊTE ===
        pdf.setFontSize(24);
        pdf.setTextColor(102, 126, 234);
        pdf.text('RAPPORTS ET ANALYTICS', margin, currentY);
        currentY += 10;

        pdf.setFontSize(18);
        pdf.text('Rapport Graphiques - Analyse des Pesees', margin, currentY);
        currentY += 15;
        
        // Date et heure
        pdf.setFontSize(12);
        pdf.setTextColor(80, 80, 80);
        const now = new Date();
        pdf.text(`Genere le ${now.toLocaleDateString('fr-FR')} a ${now.toLocaleTimeString('fr-FR', {hour: '2-digit', minute: '2-digit'})}`, margin, currentY);
        currentY += 15;
        
        // === FILTRES APPLIQUÉS ===
        pdf.setDrawColor(200, 200, 200);
        pdf.setFillColor(248, 249, 250);
        pdf.rect(margin, currentY, pageWidth - 2*margin, 30, 'FD');
        currentY += 8;
        
        pdf.setFontSize(14);
        pdf.setTextColor(0, 0, 0);
        pdf.text('Criteres de Filtrage Appliques', margin + 5, currentY);
        currentY += 8;
        
        pdf.setFontSize(10);
        const filters = reportsData.currentFilters;
        
        const periodText = filters.period === 'custom' 
            ? `Periode personnalisee: ${filters.startDate} vers ${filters.endDate}`
            : `Periode: ${filters.period}`;
        pdf.text(`- ${periodText}`, margin + 5, currentY);
        currentY += 5;
        
        if (filters.site !== 'all') {
            pdf.text(`- Site: ${filters.site}`, margin + 5, currentY);
            currentY += 5;
        }
        if (filters.client !== 'all') {
            pdf.text(`- Client: ${filters.client}`, margin + 5, currentY);
            currentY += 5;
        }
        if (filters.fournisseur !== 'all') {
            pdf.text(`- Fournisseur: ${filters.fournisseur}`, margin + 5, currentY);
            currentY += 5;
        }
        
        currentY += 15;
        
        // === MÉTRIQUES CLÉS ===
        pdf.setFontSize(16);
        pdf.setTextColor(102, 126, 234);
        pdf.text('Indicateurs Cles de Performance', margin, currentY);
        currentY += 12;
        
        const metrics = {
            totalWeighings: document.getElementById('totalWeighingsMetric')?.textContent || '0',
            totalWeight: document.getElementById('totalWeightMetric')?.textContent || '0 T',
            avgWeight: document.getElementById('avgWeightMetric')?.textContent || '0 kg',
            activeClients: document.getElementById('activeClientsMetric')?.textContent || '0',
            activeFournisseurs: document.getElementById('activeFournisseursMetric')?.textContent || '0'
        };
        
        // Métriques en tableau
        pdf.setFillColor(102, 126, 234);
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(10);
        
        const metricsData = [
            ['Indicateur', 'Valeur'],
            ['Total des pesees', metrics.totalWeighings],
            ['Poids total', metrics.totalWeight],
            ['Poids moyen', metrics.avgWeight],
            ['Clients actifs', metrics.activeClients],
            ['Fournisseurs actifs', metrics.activeFournisseurs]
        ];
        
        const colWidthsMetrics = [80, 60];
        let startX = margin;
        
        metricsData.forEach((row, rowIndex) => {
            if (rowIndex === 0) {
                // En-tête
                pdf.setFillColor(102, 126, 234);
                pdf.setTextColor(255, 255, 255);
                pdf.rect(startX, currentY, colWidthsMetrics[0] + colWidthsMetrics[1], 8, 'F');
            } else {
                // Lignes alternées
                if (rowIndex % 2 === 0) {
                    pdf.setFillColor(248, 249, 250);
                    pdf.rect(startX, currentY, colWidthsMetrics[0] + colWidthsMetrics[1], 6, 'F');
                }
                pdf.setTextColor(0, 0, 0);
            }
            
            let cellX = startX + 3;
            row.forEach((cell, cellIndex) => {
                pdf.text(cell.toString(), cellX, currentY + (rowIndex === 0 ? 6 : 4));
                cellX += colWidthsMetrics[cellIndex];
            });
            
            currentY += rowIndex === 0 ? 10 : 6;
        });
        
        currentY += 15;
        
        // === GRAPHIQUES HAUTE QUALITÉ ===
        const chartsToExport = [
            { id: 'evolutionChart', title: 'Evolution des Pesees dans le Temps', fullPage: true },
            { id: 'productsChart', title: 'Repartition par Type de Produit', fullPage: false },
            { id: 'topClientsChart', title: 'Top 10 des Clients Principaux', fullPage: false },
            { id: 'topFournisseursChart', title: 'Top 10 des Fournisseurs Principaux', fullPage: false },
            { id: 'sitesChart', title: 'Performance par Site de Pesage', fullPage: false },
            { id: 'clientFournisseurChart', title: 'Relations Client-Fournisseur', fullPage: false }
        ];
        
        for (const chartInfo of chartsToExport) {
            const chartElement = document.getElementById(chartInfo.id);
            if (!chartElement) {
                console.warn(`Graphique ${chartInfo.id} non trouvé`);
                continue;
            }
            
            try {
                // Nouvelle page si nécessaire (sauf premier graphique)
                if (chartInfo.fullPage || currentY > pageHeight - 120) {
                    pdf.addPage('portrait');
                    currentY = margin;
                }
                
                // Titre du graphique
                pdf.setFontSize(14);
                pdf.setTextColor(102, 126, 234);
                pdf.text(chartInfo.title, margin, currentY);
                currentY += 12;
                
                // Capturer le conteneur complet avec légendes
                const chartContainer = chartElement.closest('.chart-card');
                if (!chartContainer) {
                    console.warn(`Container non trouvé pour ${chartInfo.id}`);
                    continue;
                }
                
                // Configuration haute résolution pour PDF
                const canvas = await html2canvas(chartContainer, {
                    backgroundColor: '#ffffff',
                    scale: 3, // Haute résolution
                    useCORS: true,
                    allowTaint: false,
                    logging: false,
                    onclone: (clonedDoc) => {
                        // Améliorer lisibilité des légendes
                        const legends = clonedDoc.querySelectorAll('.chartjs-legend, canvas + div, .chart-legend, .legend');
                        legends.forEach(legend => {
                            legend.style.fontSize = '16px';
                            legend.style.fontWeight = 'bold';
                            legend.style.color = '#000000';
                            legend.style.background = '#ffffff';
                        });
                    }
                });
                
                const imgData = canvas.toDataURL('image/png', 0.95);
                
                // Dimensions optimales selon le type
                const maxWidth = pageWidth - 2 * margin;
                const maxHeight = chartInfo.fullPage ? 180 : 120;
                
                let imgWidth = maxWidth;
                let imgHeight = (canvas.height * imgWidth) / canvas.width;
                
                if (imgHeight > maxHeight) {
                    imgHeight = maxHeight;
                    imgWidth = (canvas.width * imgHeight) / canvas.height;
                }
                
                // Centrer le graphique
                const imgX = (pageWidth - imgWidth) / 2;
                
                // Vérifier l'espace restant
                if (currentY + imgHeight > pageHeight - margin) {
                    pdf.addPage('portrait');
                    currentY = margin;
                    
                    // Répéter le titre
                    pdf.setFontSize(14);
                    pdf.setTextColor(102, 126, 234);
                    pdf.text(chartInfo.title, margin, currentY);
                    currentY += 12;
                }
                
                pdf.addImage(imgData, 'PNG', imgX, currentY, imgWidth, imgHeight);
                currentY += imgHeight + 20;
                
                console.log(`✅ Graphique ${chartInfo.title} ajouté au PDF`);
                
            } catch (error) {
                console.warn(`Erreur capture ${chartInfo.id}:`, error);
                
                // Message d'erreur dans le PDF
                pdf.setFontSize(12);
                pdf.setTextColor(200, 0, 0);
                pdf.text(`Attention: Impossible de capturer le graphique "${chartInfo.title}"`, margin, currentY);
                currentY += 15;
            }
        }
        
        // === PIED DE PAGE ===
        addPageNumbersToDocument(pdf, 'Rapport Graphiques');
        
        // Télécharger PDF Graphiques
        const fileName = `RAPPORTS_ANALYTICS_Graphiques_${new Date().toISOString().split('T')[0]}_${Date.now()}.pdf`;
        pdf.save(fileName);
        
        showTemporaryMessage('✅ PDF Graphiques généré', 'success');
        console.log('📊 Export PDF Graphiques réussi:', fileName);
        
        return fileName;
        
    } catch (error) {
        console.error('❌ Erreur export PDF Graphiques:', error);
        showTemporaryMessage('❌ Erreur génération PDF Graphiques', 'error');
        throw error;
    }
}

/**
 * Export PDF séparé - Tableau en paysage
 */
async function exportTableauToPDF() {
    try {
        showTemporaryMessage('📋 Génération PDF Tableau...', 'warning');

        // Créer barre de progression
        const progressContainer = document.createElement('div');
        progressContainer.id = 'pdf-progress';
        progressContainer.style.cssText = `
            position: fixed; top: 50%; left: 50%; transform: translate(-50%, -50%);
            background: white; padding: 20px; border-radius: 8px; box-shadow: 0 4px 12px rgba(0,0,0,0.3);
            z-index: 10000; min-width: 300px; text-align: center;
        `;
        progressContainer.innerHTML = `
            <h3 style="margin: 0 0 15px 0; color: #333;">Génération PDF en cours...</h3>
            <div style="background: #f0f0f0; border-radius: 10px; overflow: hidden; height: 20px;">
                <div id="pdf-progress-bar" style="height: 100%; background: linear-gradient(90deg, #667eea, #764ba2); transition: width 0.3s; width: 0%;"></div>
            </div>
            <p id="pdf-progress-text" style="margin: 10px 0 0 0; color: #666; font-size: 14px;">Initialisation...</p>
        `;
        document.body.appendChild(progressContainer);

        const updateProgress = (percent, text) => {
            const bar = document.getElementById('pdf-progress-bar');
            const textEl = document.getElementById('pdf-progress-text');
            if (bar) bar.style.width = percent + '%';
            if (textEl) textEl.textContent = text;
        };

        updateProgress(10, 'Configuration PDF...');
        
        const { jsPDF } = window.jspdf;
        const pdf = new jsPDF('landscape', 'mm', 'a4'); // Format paysage forcé

        // Configuration de compression
        pdf.setProperties({
            title: 'RAPPORTS ET ANALYTICS - Rapport Détaillé',
            subject: 'Données de pesées',
            author: 'RAPPORTS ET ANALYTICS',
            creator: 'Système de pesage'
        });
        
        // Configuration UTF-8
        pdf.setFont("helvetica");
        pdf.setCharSpace(0);
        
        const pageWidth = pdf.internal.pageSize.getWidth(); // 297mm
        const pageHeight = pdf.internal.pageSize.getHeight(); // 210mm
        const margin = 15;
        let currentY = margin;
        
        // === EN-TÊTE PAYSAGE AMÉLIORÉ ===
        // Espace réservé pour logo (60x30mm)
        pdf.setDrawColor(200, 200, 200);
        pdf.setFillColor(250, 250, 250);
        pdf.rect(margin, currentY, 60, 30, 'FD');
        pdf.setFontSize(8);
        pdf.setTextColor(150, 150, 150);
        pdf.text('Logo SITE / SOCIETE', margin + 5, currentY + 18);

        // Titre principal à droite du logo
        pdf.setFontSize(20);
        pdf.setTextColor(102, 126, 234);
        pdf.text('RAPPORTS ET ANALYTICS', margin + 70, currentY + 12);

        pdf.setFontSize(16);
        pdf.text('Donnees Detaillees des Pesees', margin + 70, currentY + 25);
        currentY += 35;

        // Date et informations
        pdf.setFontSize(12);
        pdf.setTextColor(80, 80, 80);
        const now = new Date();
        pdf.text(`Genere le ${now.toLocaleDateString('fr-FR')} a ${now.toLocaleTimeString('fr-FR', {hour: '2-digit', minute: '2-digit'})}`, margin, currentY);

        // Badge "CONFIDENTIEL" en haut à droite
        pdf.setFillColor(220, 53, 69);
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(10);
        const confidText = 'CONFIDENTIEL';
        const confidWidth = pdf.getTextWidth(confidText) + 10;
        pdf.rect(pageWidth - margin - confidWidth, margin, confidWidth, 12, 'F');
        pdf.text(confidText, pageWidth - margin - confidWidth + 5, margin + 8);

        pdf.setTextColor(80, 80, 80);
        currentY += 8;
        
        // Filtres en ligne
        pdf.setFontSize(10);
        const filters = reportsData.currentFilters;
        
        let filtersText = `Filtres appliques: `;
        filtersText += filters.period === 'custom' 
            ? `Periode: ${filters.startDate} vers ${filters.endDate}` 
            : `Periode: ${filters.period}`;
        
        if (filters.site !== 'all') filtersText += ` | Site: ${filters.site}`;
        if (filters.client !== 'all') filtersText += ` | Client: ${filters.client}`;
        if (filters.fournisseur !== 'all') filtersText += ` | Fournisseur: ${filters.fournisseur}`;
        
        pdf.text(filtersText, margin, currentY);
        currentY += 15;
        
        updateProgress(25, 'Récupération des données...');

        // === RÉCUPÉRER DONNÉES ===
        const data = await loadReportsData();
        if (!data || data.length === 0) {
            pdf.setFontSize(14);
            pdf.setTextColor(150, 150, 150);
            pdf.text('Aucune donnee disponible pour la periode et les criteres selectionnes.', margin, currentY);
            
            const fileName = `RAPPORTS_ANALYTICS_Tableau_Vide_${new Date().toISOString().split('T')[0]}.pdf`;
            pdf.save(fileName);
            
            showTemporaryMessage('⚠️ PDF Tableau vide généré', 'warning');
            return fileName;
        }
        
        // === RÉSUMÉ DES DONNÉES ===
        pdf.setFontSize(12);
        pdf.setTextColor(0, 0, 0);
        const totalRecords = data.length;
        const displayedRecords = Math.min(1000, totalRecords); // Limite à 1000 lignes
        
        pdf.text(`Total des enregistrements: ${totalRecords} | Affiches: ${displayedRecords}`, margin, currentY);
        currentY += 8;

        // === LÉGENDE DES MOUVEMENTS ===
        pdf.setFontSize(10);
        pdf.setTextColor(102, 126, 234);
        pdf.text('Legende des Mouvements:', margin, currentY);
        currentY += 5;

        pdf.setFontSize(9);
        pdf.setTextColor(0, 0, 0);
        const legendItems = [
            'E = Entree | S = Sortie | T = Transit | R = Retour',
            'PE = Pesee Entree | PS = Pesee Sortie'
        ];

        legendItems.forEach(item => {
            pdf.text(`• ${item}`, margin + 5, currentY);
            currentY += 4;
        });

        currentY += 8;

        updateProgress(40, 'Préparation du tableau...');

        // === TABLEAU OPTIMISÉ PAYSAGE ===
        // Colonnes avec largeurs optimisées pour A4 paysage
        const colWidths = [32, 38, 42, 40, 40, 26, 35, 26, 24]; // Total: 303mm
        const headers = ['Date', 'Code Pesee', 'N° Ticket', 'Client', 'Fournisseur', 'Mouvement', 'Produit', 'Site', 'Poids'];
        
        // En-têtes avec style professionnel
        pdf.setFillColor(102, 126, 234);
        pdf.setTextColor(255, 255, 255);
        pdf.setFontSize(11);

        const totalWidth = colWidths.reduce((sum, width) => sum + width, 0);
        const tableStartX = (pageWidth - totalWidth) / 2; // Centrer le tableau

        // En-tête avec bordures
        pdf.setDrawColor(80, 80, 80);
        pdf.setLineWidth(0.3);
        pdf.rect(tableStartX, currentY, totalWidth, 12, 'FD');
        
        let startX = tableStartX + 8;
        headers.forEach((header, index) => {
            pdf.text(header, startX, currentY + 8);

            // Bordures verticales entre colonnes
            if (index < headers.length - 1) {
                const separatorX = tableStartX + colWidths.slice(0, index + 1).reduce((sum, width) => sum + width, 0);
                pdf.line(separatorX, currentY, separatorX, currentY + 12);
            }

            startX += colWidths[index];
        });
        currentY += 16;
        
        // === DONNÉES DU TABLEAU ===
        pdf.setTextColor(0, 0, 0);
        pdf.setFontSize(9);

        const maxRowsPerPage = Math.floor((pageHeight - currentY - 20) / 7);
        let rowCount = 0;
        let pageNum = 1;
        let totalPoids = 0; // Variable pour calculer le poids total
        
        // Trier par client pour éviter les coupures de groupes
        const sortedData = [...data].sort((a, b) => {
            const clientA = (a.NomClient || '').toLowerCase();
            const clientB = (b.NomClient || '').toLowerCase();
            if (clientA === clientB) {
                return new Date(a.dmv) - new Date(b.dmv); // Puis par date
            }
            return clientA.localeCompare(clientB);
        });

        for (let i = 0; i < Math.min(displayedRecords, sortedData.length); i++) {
            const item = sortedData[i];
            const nextItem = sortedData[i + 1];

            // Mise à jour de la progression
            if (i % 50 === 0) { // Mise à jour toutes les 50 lignes
                const progress = 50 + (i / Math.min(displayedRecords, sortedData.length)) * 40;
                updateProgress(progress, `Traitement ligne ${i + 1}/${Math.min(displayedRecords, sortedData.length)}...`);
                await new Promise(resolve => setTimeout(resolve, 1)); // Micro-pause pour UI
            }

            // Ajouter le poids au total
            const poids = parseFloat(item.PoidsNet || 0);
            totalPoids += poids;

            // Pagination intelligente : éviter de séparer le même client
            const isLastOfClient = !nextItem || (nextItem.NomClient !== item.NomClient);
            const spaceNeeded = isLastOfClient ? 1 : 2; // Plus d'espace si fin de groupe

            // Nouvelle page si nécessaire (avec logique intelligente)
            if (rowCount >= maxRowsPerPage - spaceNeeded) {
                // Numéro de page
                pdf.setFontSize(8);
                pdf.setTextColor(120, 120, 120);
                pdf.text(`Page ${pageNum}`, pageWidth - 30, pageHeight - 10);
                
                pdf.addPage('landscape');
                pageNum++;
                currentY = margin;
                
                // Répéter en-têtes
                pdf.setFillColor(102, 126, 234);
                pdf.setTextColor(255, 255, 255);
                pdf.setFontSize(11);
                pdf.rect(tableStartX, currentY, totalWidth, 12, 'F');
                
                startX = tableStartX + 8;
                headers.forEach((header, index) => {
                    pdf.text(header, startX, currentY + 8);
                    startX += colWidths[index];
                });
                currentY += 16;
                
                pdf.setTextColor(0, 0, 0);
                pdf.setFontSize(9);
                rowCount = 0;
            }
            
            // Alternance de couleurs avec bordures
            pdf.setDrawColor(220, 220, 220);
            pdf.setLineWidth(0.1);

            if (rowCount % 2 === 0) {
                pdf.setFillColor(248, 249, 250);
                pdf.rect(tableStartX, currentY - 1, totalWidth, 7, 'FD');
            } else {
                pdf.rect(tableStartX, currentY - 1, totalWidth, 7, 'D');
            }
            
            // Données de la ligne
            const rowData = [
                new Date(item.dmv).toLocaleDateString('fr-FR', {day: '2-digit', month: '2-digit', year: 'numeric'}),
                item.CodePesee || '-',
                item.NumTicket || '-',
                item.NomClient || '-',
                item.NomFournisseur || '-',
                item.Mouvement || '-',
                item.NomProduit || '-',
                item.NomSite || item.CodeSite || '-',
                (() => {
                    const poids = parseFloat(item.PoidsNet || 0);
                    if (poids >= 1000) {
                        const tonnes = poids / 1000;
                        return `${Number.isInteger(tonnes) ? tonnes.toString() : tonnes.toFixed(1)} t`;
                    } else {
                        return `${Number.isInteger(poids) ? poids.toString() : poids.toFixed(1)} kg`;
                    }
                })()
            ];
            
            startX = tableStartX + 8;
            rowData.forEach((cellData, index) => {
                let text = cellData.toString();

                // Gestion spéciale de la taille de police pour les champs longs
                const isLongField = index === 1 || index === 2; // Code Pesee et N° Ticket
                const isSiteField = index === 7; // Site
                const maxLengthsSmallFont = [10, 25, 25, 16, 16, 10, 14, 14, 8]; // Avec police réduite
                const maxLengthsNormalFont = [10, 15, 15, 16, 16, 10, 14, 10, 8]; // Avec police normale

                let fontSize = 9; // Taille par défaut
                let maxLen = maxLengthsNormalFont[index];

                // Si c'est un champ long (Code Pesee ou N° Ticket) et qu'il est long
                if (isLongField && text.length > 15) {
                    fontSize = 7; // Police plus petite
                    maxLen = maxLengthsSmallFont[index];
                }

                // Si c'est le champ Site, toujours utiliser une police de 7 points
                if (isSiteField) {
                    fontSize = 7;
                    maxLen = maxLengthsSmallFont[index];
                }

                // Troncature seulement si vraiment nécessaire
                if (text.length > maxLen) {
                    text = text.substring(0, maxLen - 2) + '..';
                }

                // Appliquer la taille de police
                pdf.setFontSize(fontSize);
                pdf.text(text, startX, currentY + 5);

                // Remettre la taille normale pour les autres cellules
                if (fontSize !== 9) {
                    pdf.setFontSize(9);
                }

                // Bordures verticales entre colonnes pour les données
                if (index < rowData.length - 1) {
                    const separatorX = tableStartX + colWidths.slice(0, index + 1).reduce((sum, width) => sum + width, 0);
                    pdf.setDrawColor(220, 220, 220);
                    pdf.line(separatorX, currentY - 1, separatorX, currentY + 6);
                }

                startX += colWidths[index];
            });
            
            currentY += 7;
            rowCount++;
        }

        // === TOTAL POIDS NET SOUS LA COLONNE ===
        // Calculer la position - étendre sur les 2 dernières colonnes (Site + Poids) pour plus d'espace
        const totalColumnX = tableStartX + colWidths.slice(0, -2).reduce((sum, width) => sum + width, 0) + 8;
        const totalAreaWidth = colWidths[colWidths.length - 2] + colWidths[colWidths.length - 1]; // Site + Poids

        // Formatage du poids total (format plus court)
        const poidsTotal = (() => {
            if (totalPoids >= 1000) {
                const tonnes = totalPoids / 1000;
                const tonnesStr = Number.isInteger(tonnes) ? tonnes.toString() : tonnes.toFixed(2);
                return `${tonnesStr}T`; // Format compact sans espace
            } else {
                const kgStr = Number.isInteger(totalPoids) ? totalPoids.toString() : totalPoids.toFixed(1);
                return `${kgStr}kg`; // Format compact sans espace
            }
        })();

        // Ligne de séparation au-dessus du total (sur les 2 dernières colonnes)
        pdf.setDrawColor(102, 126, 234);
        pdf.setLineWidth(0.5);
        pdf.line(totalColumnX, currentY + 2, totalColumnX + totalAreaWidth - 8, currentY + 2);

        // Afficher le total aligné à droite dans l'espace disponible
        currentY += 8;
        pdf.setFontSize(9);
        pdf.setTextColor(102, 126, 234);
        pdf.setFont("helvetica", "bold");

        const totalText = `TOTAL: ${poidsTotal}`;
        const totalTextWidth = pdf.getTextWidth(totalText);

        // Calculer position avec plus de marge pour éviter la coupure
        const availableWidth = totalAreaWidth - 16; // Plus de marge
        const alignedX = Math.max(totalColumnX + 4, totalColumnX + availableWidth - totalTextWidth);

        // S'assurer que le texte reste dans les limites de la page
        const maxX = pageWidth - margin - totalTextWidth;
        const finalX = Math.min(alignedX, maxX);

        pdf.text(totalText, finalX, currentY);
        pdf.setFont("helvetica", "normal");

        currentY += 10;

        // === PIED DE DERNIÈRE PAGE ===
        if (totalRecords > displayedRecords) {
            currentY += 10;
            pdf.setFontSize(10);
            pdf.setTextColor(150, 150, 150);
            pdf.text(`Note: Seules les ${displayedRecords} premieres lignes sont affichees sur ${totalRecords} au total.`, margin, currentY);
            currentY += 10;
        }

        // === STATISTIQUES AVANCÉES ===
        currentY += 15;

        // Calculer statistiques
        const sortedByWeight = [...data].sort((a, b) => (parseFloat(b.PoidsNet) || 0) - (parseFloat(a.PoidsNet) || 0));
        const top5 = sortedByWeight.slice(0, 5);

        // Répartition par mouvement
        const mouvementStats = data.reduce((acc, item) => {
            const mouv = item.Mouvement || 'Non défini';
            acc[mouv] = (acc[mouv] || 0) + 1;
            return acc;
        }, {});

        // Moyenne par jour
        const dateGroups = data.reduce((acc, item) => {
            const date = new Date(item.dmv).toDateString();
            if (!acc[date]) acc[date] = [];
            acc[date].push(parseFloat(item.PoidsNet) || 0);
            return acc;
        }, {});

        const dailyAverages = Object.entries(dateGroups).map(([date, weights]) => ({
            date: new Date(date).toLocaleDateString('fr-FR'),
            average: weights.reduce((sum, w) => sum + w, 0) / weights.length,
            count: weights.length
        }));

        // === TOP 5 PESÉES ===
        pdf.setFontSize(14);
        pdf.setTextColor(102, 126, 234);
        pdf.text('Top 5 des Plus Grosses Pesees (Global)', margin, currentY);
        currentY += 8;

        pdf.setFontSize(9);
        pdf.setTextColor(0, 0, 0);
        top5.forEach((item, index) => {
            const poids = parseFloat(item.PoidsNet || 0);
            const poidsStr = poids >= 1000 ? `${(poids/1000).toFixed(1)} T` : `${poids} kg`;
            const ligne = `${index + 1}. ${poidsStr} - ${item.NomClient || 'N/A'} (${new Date(item.dmv).toLocaleDateString('fr-FR')})`;
            pdf.text(`• ${ligne}`, margin + 5, currentY);
            currentY += 4;
        });

        currentY += 8;

        // === TOP 5 PESÉES PAR FOURNISSEUR ===
        const groupedByFournisseur = sortedByWeight.reduce((acc, item) => {
            const fournisseur = item.NomFournisseur || 'Non défini';
            if (!acc[fournisseur]) acc[fournisseur] = [];
            acc[fournisseur].push(item);
            return acc;
        }, {});

        const top5ParFournisseur = Object.entries(groupedByFournisseur)
            .map(([fournisseur, items]) => ({
                fournisseur,
                topItem: items[0], // Le plus lourd (déjà trié)
                totalPesees: items.length,
                poidsTotal: items.reduce((sum, item) => sum + parseFloat(item.PoidsNet || 0), 0)
            }))
            .sort((a, b) => parseFloat(b.topItem.PoidsNet || 0) - parseFloat(a.topItem.PoidsNet || 0))
            .slice(0, 5);

        pdf.setFontSize(14);
        pdf.setTextColor(102, 126, 234);
        pdf.text('Top 5 des Plus Grosses Pesees par Fournisseur', margin, currentY);
        currentY += 8;

        pdf.setFontSize(9);
        pdf.setTextColor(0, 0, 0);
        top5ParFournisseur.forEach((fournisseurData, index) => {
            const poids = parseFloat(fournisseurData.topItem.PoidsNet || 0);
            const poidsStr = poids >= 1000 ? `${(poids/1000).toFixed(1)} T` : `${poids} kg`;
            const totalFournisseurStr = fournisseurData.poidsTotal >= 1000 ?
                `${(fournisseurData.poidsTotal/1000).toFixed(1)} T` :
                `${fournisseurData.poidsTotal.toFixed(1)} kg`;

            const ligne = `${index + 1}. ${fournisseurData.fournisseur}: ${poidsStr} (Total: ${totalFournisseurStr}, ${fournisseurData.totalPesees} pesees)`;
            pdf.text(`• ${ligne}`, margin + 5, currentY);
            currentY += 4;
        });

        currentY += 8;

        // === RÉPARTITION PAR MOUVEMENT ===
        pdf.setFontSize(14);
        pdf.setTextColor(102, 126, 234);
        pdf.text('Repartition par Mouvement', margin, currentY);
        currentY += 8;

        pdf.setFontSize(9);
        pdf.setTextColor(0, 0, 0);
        const totalMouvements = Object.values(mouvementStats).reduce((sum, count) => sum + count, 0);

        Object.entries(mouvementStats).forEach(([mouvement, count]) => {
            const percentage = ((count / totalMouvements) * 100).toFixed(1);
            pdf.text(`• ${mouvement}: ${count} pesees (${percentage}%)`, margin + 5, currentY);
            currentY += 4;
        });

        currentY += 8;

        // === MOYENNES QUOTIDIENNES AVEC TENDANCES ===
        if (dailyAverages.length > 1) {
            pdf.setFontSize(14);
            pdf.setTextColor(102, 126, 234);
            pdf.text('Moyennes par Jour avec Tendances', margin, currentY);
            currentY += 8;

            pdf.setFontSize(9);
            pdf.setTextColor(0, 0, 0);
            dailyAverages.slice(-5).forEach((day, index, arr) => {
                const avgStr = day.average >= 1000 ? `${(day.average/1000).toFixed(1)} T` : `${day.average.toFixed(1)} kg`;

                // Calculer la tendance
                let trendIcon = '';
                let trendColor = [0, 0, 0]; // Noir par défaut
                if (index > 0) {
                    const prevAvg = arr[index - 1].average;
                    const change = ((day.average - prevAvg) / prevAvg * 100);
                    if (change > 5) {
                        trendIcon = ' [+]';
                        trendColor = [40, 167, 69]; // Vert
                    } else if (change < -5) {
                        trendIcon = ' [-]';
                        trendColor = [220, 53, 69]; // Rouge
                    } else {
                        trendIcon = ' [=]';
                        trendColor = [255, 193, 7]; // Jaune
                    }
                }

                pdf.setTextColor(...trendColor);
                pdf.text(`• ${day.date}: ${avgStr} (${day.count} pesees)${trendIcon}`, margin + 5, currentY);
                pdf.setTextColor(0, 0, 0); // Remettre en noir
                currentY += 4;
            });
            currentY += 8;

            // Légende des tendances
            pdf.setFontSize(8);
            pdf.setTextColor(120, 120, 120);
            pdf.text('Legende: [+] Hausse >5% | [-] Baisse >5% | [=] Stable +/-5%', margin + 5, currentY);
            currentY += 8;
        }

        // === TENDANCES PAR FOURNISSEUR ===
        // Calculer moyennes par fournisseur par période
        const fournisseurDateGroups = sortedData.reduce((acc, item) => {
            const fournisseur = item.NomFournisseur || 'Non défini';
            const date = new Date(item.dmv).toDateString();
            const key = `${fournisseur}|${date}`;

            if (!acc[key]) {
                acc[key] = { fournisseur, date, weights: [], total: 0, count: 0 };
            }

            const poids = parseFloat(item.PoidsNet || 0);
            acc[key].weights.push(poids);
            acc[key].total += poids;
            acc[key].count++;

            return acc;
        }, {});

        const fournisseurAverages = Object.values(fournisseurDateGroups)
            .map(group => ({
                fournisseur: group.fournisseur,
                date: new Date(group.date).toLocaleDateString('fr-FR'),
                dateObj: new Date(group.date),
                average: group.total / group.count,
                count: group.count
            }))
            .sort((a, b) => a.dateObj - b.dateObj);

        // Grouper par fournisseur et calculer tendances
        const fournisseurTrends = fournisseurAverages.reduce((acc, item) => {
            if (!acc[item.fournisseur]) acc[item.fournisseur] = [];
            acc[item.fournisseur].push(item);
            return acc;
        }, {});

        const topFournisseurTrends = Object.entries(fournisseurTrends)
            .filter(([fournisseur, data]) => data.length >= 2) // Au moins 2 points pour calculer tendance
            .map(([fournisseur, data]) => {
                const latest = data[data.length - 1];
                const previous = data[data.length - 2];
                const change = ((latest.average - previous.average) / previous.average * 100);

                let trendIcon = '[=]';
                let trendColor = [255, 193, 7]; // Jaune par défaut

                if (change > 10) {
                    trendIcon = '[++]';
                    trendColor = [40, 167, 69]; // Vert foncé
                } else if (change > 5) {
                    trendIcon = '[+]';
                    trendColor = [40, 167, 69]; // Vert
                } else if (change < -10) {
                    trendIcon = '[--]';
                    trendColor = [220, 53, 69]; // Rouge foncé
                } else if (change < -5) {
                    trendIcon = '[-]';
                    trendColor = [220, 53, 69]; // Rouge
                }

                return {
                    fournisseur,
                    latest,
                    change: change.toFixed(1),
                    trendIcon,
                    trendColor,
                    totalPesees: data.reduce((sum, d) => sum + d.count, 0)
                };
            })
            .sort((a, b) => Math.abs(parseFloat(b.change)) - Math.abs(parseFloat(a.change))) // Trier par amplitude de changement
            .slice(0, 5); // Top 5

        if (topFournisseurTrends.length > 0) {
            pdf.setFontSize(14);
            pdf.setTextColor(102, 126, 234);
            pdf.text('Tendances par Fournisseur (Evolution Recent)', margin, currentY);
            currentY += 8;

            pdf.setFontSize(9);
            topFournisseurTrends.forEach(trend => {
                const avgStr = trend.latest.average >= 1000 ?
                    `${(trend.latest.average/1000).toFixed(1)} T` :
                    `${trend.latest.average.toFixed(1)} kg`;

                pdf.setTextColor(...trend.trendColor);
                pdf.text(`• ${trend.fournisseur}: ${avgStr} ${trend.trendIcon} (${trend.change}%)`, margin + 5, currentY);
                pdf.setTextColor(0, 0, 0);
                currentY += 4;
            });
            currentY += 8;

            // Légende spécifique fournisseurs
            pdf.setFontSize(8);
            pdf.setTextColor(120, 120, 120);
            pdf.text('Legende: [++] Forte hausse >10% | [+] Hausse >5% | [-] Baisse >5% | [--] Forte baisse >10%', margin + 5, currentY);
            currentY += 8;
        }

        // === ANALYSE TEMPORELLE PAR CLIENT ===
        const clientTimingData = sortedData.reduce((acc, item) => {
            const client = item.NomClient || 'Non défini';
            if (!acc[client]) acc[client] = [];
            acc[client].push(new Date(item.dmv));
            return acc;
        }, {});

        const clientIntervals = Object.entries(clientTimingData)
            .filter(([client, dates]) => dates.length > 1)
            .map(([client, dates]) => {
                dates.sort((a, b) => a - b);
                const intervals = [];
                for (let i = 1; i < dates.length; i++) {
                    const diffDays = (dates[i] - dates[i-1]) / (1000 * 60 * 60 * 24);
                    intervals.push(diffDays);
                }
                const avgInterval = intervals.reduce((sum, interval) => sum + interval, 0) / intervals.length;
                return { client, avgInterval: Math.round(avgInterval * 10) / 10, peseeCount: dates.length };
            })
            .sort((a, b) => a.avgInterval - b.avgInterval)
            .slice(0, 5);

        if (clientIntervals.length > 0) {
            pdf.setFontSize(14);
            pdf.setTextColor(102, 126, 234);
            pdf.text('Frequence des Pesees par Client', margin, currentY);
            currentY += 8;

            pdf.setFontSize(9);
            pdf.setTextColor(0, 0, 0);
            clientIntervals.forEach(({client, avgInterval, peseeCount}) => {
                const intervalText = avgInterval < 1 ?
                    `${(avgInterval * 24).toFixed(1)}h` :
                    `${avgInterval.toFixed(1)} jours`;
                pdf.text(`• ${client}: ${intervalText} en moyenne (${peseeCount} pesees)`, margin + 5, currentY);
                currentY += 4;
            });
            currentY += 8;
        }

        // === ALERTES PESÉES INHABITUELLES ===
        const avgWeight = totalPoids / sortedData.length;
        const weightThreshold = avgWeight * 3; // 3x la moyenne = inhabituel
        const unusualWeighings = sortedData
            .filter(item => parseFloat(item.PoidsNet || 0) > weightThreshold)
            .slice(0, 3);

        if (unusualWeighings.length > 0) {
            pdf.setFontSize(14);
            pdf.setTextColor(220, 53, 69); // Rouge pour alertes
            pdf.text('ALERTE: Pesees Inhabituelles', margin, currentY);
            currentY += 8;

            pdf.setFontSize(9);
            pdf.setTextColor(0, 0, 0);
            unusualWeighings.forEach(item => {
                const poids = parseFloat(item.PoidsNet || 0);
                const poidsStr = poids >= 1000 ? `${(poids/1000).toFixed(1)} T` : `${poids} kg`;
                const ratio = (poids / avgWeight).toFixed(1);
                pdf.text(`• ${poidsStr} (${ratio}x la moyenne) - ${item.NomClient || 'N/A'} (${new Date(item.dmv).toLocaleDateString('fr-FR')})`, margin + 5, currentY);
                currentY += 4;
            });
            currentY += 8;
        }


        // Numéro de dernière page
        pdf.setFontSize(8);
        pdf.setTextColor(120, 120, 120);
        pdf.text(`Page ${pageNum}`, pageWidth - 30, pageHeight - 10);

        updateProgress(95, 'Finalisation du document...');

        // Ajouter watermark sur toutes les pages
        addWatermarkToDocument(pdf);

        updateProgress(100, 'Téléchargement en cours...');

        // Télécharger PDF Tableau
        const fileName = `RAPPORTS_ANALYTICS_Tableau_${new Date().toISOString().split('T')[0]}_${Date.now()}.pdf`;
        pdf.save(fileName);
        
        // Supprimer la barre de progression
        setTimeout(() => {
            const progressEl = document.getElementById('pdf-progress');
            if (progressEl) progressEl.remove();
        }, 1000);

        showTemporaryMessage('✅ PDF Tableau généré', 'success');
        console.log('📋 Export PDF Tableau réussi:', fileName);

        return fileName;
        
    } catch (error) {
        // Supprimer la barre de progression en cas d'erreur
        const progressEl = document.getElementById('pdf-progress');
        if (progressEl) progressEl.remove();

        console.error('❌ Erreur export PDF Tableau:', error);
        showTemporaryMessage('❌ Erreur génération PDF Tableau', 'error');
        throw error;
    }
}

/**
 * Export PDF complet - Les deux fichiers
 */
async function exportBothPDFs() {
    try {
        showTemporaryMessage('📁 Génération des deux PDF...', 'info');
        
        // Générer les deux PDF en parallèle
        const [graphiquesFile, tableauFile] = await Promise.all([
            exportGraphiquesToPDF(),
            exportTableauToPDF()
        ]);
        
        showTemporaryMessage('✅ Deux PDF générés avec succès', 'success');
        console.log('📁 Export complet réussi:', { graphiquesFile, tableauFile });
        
        return { graphiquesFile, tableauFile };
        
    } catch (error) {
        console.error('❌ Erreur export PDF complet:', error);
        showTemporaryMessage('❌ Erreur génération PDF complet', 'error');
    }
}

/**
 * Ajouter un watermark sur toutes les pages
 */
function addWatermarkToDocument(pdf) {
    const totalPages = pdf.internal.getNumberOfPages();

    for (let i = 1; i <= totalPages; i++) {
        pdf.setPage(i);

        const pageWidth = pdf.internal.pageSize.getWidth();
        const pageHeight = pdf.internal.pageSize.getHeight();

        // Watermark en diagonal
        pdf.setTextColor(230, 230, 230);
        pdf.setFontSize(60);
        pdf.setFont("helvetica", "bold");

        // Calculer position centrale
        const watermarkText = 'RAPPORTS ET ANALYTICS';
        const textWidth = pdf.getTextWidth(watermarkText);

        // Rotation et positionnement
        pdf.saveGraphicsState();
        pdf.setGState(new pdf.GState({opacity: 0.1}));

        // Rotation de 45 degrés
        const centerX = pageWidth / 2;
        const centerY = pageHeight / 2;

        pdf.text(watermarkText, centerX - textWidth/2, centerY, {
            angle: 45,
            align: 'center'
        });

        pdf.restoreGraphicsState();
    }
}

/**
 * Ajouter numéros de page à un document
 */
function addPageNumbersToDocument(pdf, documentTitle) {
    const totalPages = pdf.internal.getNumberOfPages();
    
    for (let i = 1; i <= totalPages; i++) {
        pdf.setPage(i);
        pdf.setFontSize(8);
        pdf.setTextColor(120, 120, 120);
        
        // Ligne de séparation
        const pageWidth = pdf.internal.pageSize.getWidth();
        const pageHeight = pdf.internal.pageSize.getHeight();
        
        pdf.setDrawColor(220, 220, 220);
        pdf.line(15, pageHeight - 15, pageWidth - 15, pageHeight - 15);
        
        // Titre et numéro de page
        pdf.text(documentTitle, 15, pageHeight - 8);
        
        const pageText = `Page ${i} / ${totalPages}`;
        const textWidth = pdf.getTextWidth(pageText);
        pdf.text(pageText, pageWidth - 15 - textWidth, pageHeight - 8);
    }
}

// Export pour utilisation globale
if (typeof window !== 'undefined') {
    window.exportGraphiquesToPDF = exportGraphiquesToPDF;
    window.exportTableauToPDF = exportTableauToPDF;
    window.exportBothPDFs = exportBothPDFs;
}