-- ===============================================
-- Script de migration pour la compartimentalisation par sites
-- Version: 1.0
-- Date: 2025-10-08
-- ===============================================

-- IMPORTANT: Faites une sauvegarde de votre base de données avant d'exécuter ce script !
-- mysqldump -u kombarferd_dbadmin -p kombarferd_gestpeseedb > backup_avant_migration_sites.sql

-- ===============================================
-- ÉTAPE 1: Vérifier l'état actuel de la table user
-- ===============================================

-- Afficher la structure actuelle
DESCRIBE `user`;

-- Vérifier les données existantes dans les colonnes de sites
SELECT
    NUt,
    NomUt,
    CodeSite,
    NomSite,
    sites_acces,
    CASE
        WHEN sites_acces IS NOT NULL AND sites_acces != '[]' THEN 'Sites JSON définis'
        WHEN CodeSite IS NOT NULL AND CodeSite != '' THEN 'Sites texte définis'
        ELSE 'Aucun site'
    END as etat_sites
FROM `user`
LIMIT 10;

-- ===============================================
-- ÉTAPE 2: Migration des données existantes
-- ===============================================

-- Migrer les données de CodeSite et NomSite vers sites_acces (JSON)
-- Uniquement pour les utilisateurs qui n'ont pas déjà sites_acces défini

UPDATE `user`
SET sites_acces = JSON_ARRAY(
    JSON_OBJECT(
        'code', TRIM(CodeSite),
        'nom', TRIM(NomSite),
        'role', 'lecteur'
    )
)
WHERE (sites_acces IS NULL OR sites_acces = '' OR sites_acces = '[]')
  AND CodeSite IS NOT NULL
  AND CodeSite != ''
  AND NomSite IS NOT NULL
  AND NomSite != '';

-- Pour les utilisateurs avec plusieurs sites (séparés par virgule)
-- Note: Cette requête nécessite un traitement plus complexe si vous avez des données avec virgules
-- Exemple de données: CodeSite = 'SITE01,SITE02', NomSite = 'Site 1,Site 2'

-- Vérifier combien d'utilisateurs ont plusieurs sites séparés par virgule
SELECT
    COUNT(*) as users_avec_multisites,
    GROUP_CONCAT(NomUt SEPARATOR ', ') as liste_utilisateurs
FROM `user`
WHERE CodeSite LIKE '%,%'
   OR NomSite LIKE '%,%';

-- Si vous avez des données avec virgules, vous devrez les traiter manuellement
-- Exemple pour un utilisateur spécifique:
-- UPDATE `user`
-- SET sites_acces = '[{"code":"SITE01","nom":"Site 1","role":"lecteur"},{"code":"SITE02","nom":"Site 2","role":"lecteur"}]'
-- WHERE NUt = 1;

-- ===============================================
-- ÉTAPE 3: Nettoyer et normaliser la structure
-- ===============================================

-- Option A: Supprimer les colonnes redondantes (RECOMMANDÉ)
-- Attention: Cette action est irréversible !

-- ALTER TABLE `user` DROP COLUMN `CodeSite`;
-- ALTER TABLE `user` DROP COLUMN `NomSite`;

-- Option B: Garder les colonnes mais les vider (pour transition progressive)
-- Décommenter si vous préférez cette approche:

-- UPDATE `user` SET CodeSite = NULL, NomSite = NULL WHERE sites_acces IS NOT NULL;

-- ===============================================
-- ÉTAPE 4: Créer la table user_sites (alternative recommandée)
-- ===============================================

-- Cette approche est plus normalisée et flexible
-- Elle permet une meilleure gestion des relations many-to-many

CREATE TABLE IF NOT EXISTS `user_sites` (
  `id` INT NOT NULL AUTO_INCREMENT,
  `NUt` INT NOT NULL COMMENT 'ID utilisateur',
  `code_site` VARCHAR(50) NOT NULL COMMENT 'Code du site',
  `nom_site` VARCHAR(255) NOT NULL COMMENT 'Nom du site',
  `role` ENUM('lecteur', 'editeur', 'admin') DEFAULT 'lecteur' COMMENT 'Rôle sur ce site',
  `date_assignation` TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT 'Date d\'assignation',
  `actif` TINYINT(1) DEFAULT 1 COMMENT 'Site actif pour cet utilisateur',
  PRIMARY KEY (`id`),
  UNIQUE KEY `user_site_unique` (`NUt`, `code_site`),
  FOREIGN KEY (`NUt`) REFERENCES `user`(`NUt`) ON DELETE CASCADE,
  INDEX `idx_user_id` (`NUt`),
  INDEX `idx_code_site` (`code_site`),
  INDEX `idx_actif` (`actif`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci
COMMENT='Table de liaison entre utilisateurs et sites';

-- Migrer les données de sites_acces vers user_sites
-- Note: Cette requête nécessite un script PHP ou une procédure stockée pour parser le JSON
-- Voir le fichier migration_user_sites.php pour cette opération

-- ===============================================
-- ÉTAPE 5: Ajouter des index pour optimiser les performances
-- ===============================================

-- Index sur la colonne sites_acces pour les recherches JSON (MySQL 5.7+)
-- ALTER TABLE `user` ADD INDEX `idx_sites_acces` ((CAST(sites_acces AS CHAR(255))));

-- ===============================================
-- ÉTAPE 6: Vérification post-migration
-- ===============================================

-- Vérifier que tous les utilisateurs actifs ont des sites définis ou sont configurés pour l'accès complet
SELECT
    NUt,
    NomUt,
    LeGroupe,
    CASE
        WHEN sites_acces IS NULL OR sites_acces = '[]' OR sites_acces = '' THEN 'Accès illimité'
        ELSE JSON_LENGTH(sites_acces)
    END as nombre_sites,
    sites_acces
FROM `user`
WHERE actif = 1
ORDER BY NomUt;

-- Statistiques de migration
SELECT
    COUNT(*) as total_utilisateurs_actifs,
    SUM(CASE WHEN sites_acces IS NULL OR sites_acces = '[]' THEN 1 ELSE 0 END) as acces_illimite,
    SUM(CASE WHEN sites_acces IS NOT NULL AND sites_acces != '[]' THEN 1 ELSE 0 END) as acces_restreint,
    SUM(CASE WHEN JSON_LENGTH(sites_acces) = 1 THEN 1 ELSE 0 END) as un_seul_site,
    SUM(CASE WHEN JSON_LENGTH(sites_acces) > 1 THEN 1 ELSE 0 END) as plusieurs_sites
FROM `user`
WHERE actif = 1;

-- ===============================================
-- ÉTAPE 7: Requêtes utiles pour la gestion
-- ===============================================

-- Lister tous les sites disponibles dans la base
SELECT DISTINCT
    CodeSite as code,
    NomSite as nom,
    COUNT(*) as nombre_pesees
FROM pesee
WHERE CodeSite IS NOT NULL AND CodeSite != ''
GROUP BY CodeSite, NomSite
ORDER BY nombre_pesees DESC;

-- Exemple d'assignation de sites à un utilisateur
-- UPDATE `user`
-- SET sites_acces = '[
--     {"code":"SITE01","nom":"Site Principal","role":"lecteur"},
--     {"code":"SITE02","nom":"Site Secondaire","role":"editeur"}
-- ]'
-- WHERE NUt = 1;

-- Exemple de requête pour filtrer les pesées par sites d'un utilisateur
-- SET @user_id = 1;
-- SELECT p.*
-- FROM pesee p
-- WHERE p.CodeSite IN (
--     SELECT JSON_UNQUOTE(JSON_EXTRACT(sites_acces, CONCAT('$[', idx, '].code')))
--     FROM user,
--          JSON_TABLE(sites_acces, '$[*]' COLUMNS(idx FOR ORDINALITY)) AS jt
--     WHERE NUt = @user_id
-- );

-- ===============================================
-- ÉTAPE 8: Rollback (en cas de problème)
-- ===============================================

-- Si vous devez annuler la migration, restaurez votre sauvegarde:
-- mysql -u kombarferd_dbadmin -p kombarferd_gestpeseedb < backup_avant_migration_sites.sql

-- Ou si vous avez gardé les colonnes CodeSite et NomSite:
-- UPDATE `user` SET sites_acces = NULL;
-- (puis restaurez les valeurs originales de CodeSite et NomSite)

-- ===============================================
-- NOTES IMPORTANTES
-- ===============================================

/*
1. SAUVEGARDE: Faites TOUJOURS une sauvegarde complète avant d'exécuter ce script

2. ÉTAPES RECOMMANDÉES:
   a. Tester d'abord sur une copie de la base de données
   b. Exécuter les requêtes de vérification (ÉTAPE 1)
   c. Exécuter la migration des données (ÉTAPE 2)
   d. Vérifier les résultats (ÉTAPE 6)
   e. Seulement après validation, supprimer les anciennes colonnes (ÉTAPE 3)

3. FORMAT JSON attendu pour sites_acces:
   [
     {"code": "SITE01", "nom": "Site Principal", "role": "lecteur"},
     {"code": "SITE02", "nom": "Site Secondaire", "role": "editeur"}
   ]

4. Si sites_acces est NULL ou vide [], l'utilisateur a accès à TOUS les sites

5. Les rôles possibles: 'lecteur', 'editeur', 'admin'
   (à implémenter selon vos besoins)

6. Pour des données complexes avec virgules, utilisez le script PHP fourni
   (migration_user_sites_data.php)
*/
