# 🔐 Guide de Compartimentalisation par Sites

## Vue d'ensemble

Ce guide documente la mise en place de la compartimentalisation des données par sites dans votre application PWA de gestion des pesées. Chaque utilisateur ne pourra accéder qu'aux données des sites qui lui sont assignés.

---

## 📋 Table des matières

1. [Architecture](#architecture)
2. [Installation](#installation)
3. [Configuration](#configuration)
4. [Utilisation](#utilisation)
5. [API Reference](#api-reference)
6. [Exemples](#exemples)
7. [Dépannage](#dépannage)

---

## Architecture

### Composants créés

#### Backend (PHP)

| Fichier | Description |
|---------|-------------|
| `SiteAccessControl.php` | Classe principale de gestion des accès par sites |
| `sites.php` | API REST pour la gestion des affectations de sites |
| `pesees_secured.php` | Version sécurisée de l'API pesées avec filtrage automatique |
| `migration_sites.sql` | Script SQL de migration de la base de données |

#### Frontend (JavaScript/HTML)

| Fichier | Description |
|---------|-------------|
| `site-filter-client.js` | Module JavaScript de filtrage côté client |
| `admin_sites.html` | Interface d'administration pour gérer les sites des utilisateurs |

### Modifications apportées

- ✅ `auth.php` - Retourne maintenant les sites autorisés lors du login
- ✅ Table `user` - Utilise le champ JSON `sites_acces` pour stocker les sites
- ✅ Système d'authentification JWT compatible avec le filtrage par sites

---

## Installation

### Étape 1: Sauvegarde

**⚠️ IMPORTANT: Sauvegardez votre base de données avant toute modification!**

```bash
# Depuis votre terminal/ligne de commande
mysqldump -u kombarferd_dbadmin -p kombarferd_gestpeseedb > backup_$(date +%Y%m%d_%H%M%S).sql
```

### Étape 2: Migration de la base de données

1. Connectez-vous à phpMyAdmin ou votre client MySQL

2. Exécutez le script de migration:
   ```sql
   -- Ouvrir et exécuter: migration_sites.sql
   ```

3. Vérifiez la structure de la table `user`:
   ```sql
   DESCRIBE user;
   -- Devrait montrer la colonne sites_acces (type TEXT)
   ```

4. **(Optionnel)** Supprimer les anciennes colonnes redondantes:
   ```sql
   ALTER TABLE user DROP COLUMN CodeSite;
   ALTER TABLE user DROP COLUMN NomSite;
   ```

### Étape 3: Télécharger les nouveaux fichiers

Tous les fichiers ont été créés dans `pesees-api/`:

```
pesees-api/
├── SiteAccessControl.php        ← Nouveau
├── sites.php                    ← Nouveau
├── pesees_secured.php           ← Nouveau
├── site-filter-client.js        ← Nouveau
├── admin_sites.html             ← Nouveau
├── migration_sites.sql          ← Nouveau
└── auth.php                     ← Modifié
```

### Étape 4: Tester l'installation

```bash
# Tester l'API des sites
curl -X GET "https://votre-domaine.com/pesees-api/sites.php?action=list" \
     -H "Authorization: Bearer VOTRE_TOKEN"
```

---

## Configuration

### 1. Configuration des utilisateurs

#### Via l'interface admin

1. Ouvrez `admin_sites.html` dans votre navigateur
2. Connectez-vous avec un compte administrateur
3. Sélectionnez un utilisateur
4. Cochez les sites auxquels il doit avoir accès
5. Cliquez sur "Enregistrer"

#### Via SQL direct

```sql
-- Assigner des sites à un utilisateur
UPDATE user
SET sites_acces = '[
    {"code": "SITE01", "nom": "Site Principal", "role": "lecteur"},
    {"code": "SITE02", "nom": "Site Secondaire", "role": "lecteur"}
]'
WHERE NUt = 1;

-- Retirer tous les sites (accès illimité)
UPDATE user SET sites_acces = NULL WHERE NUt = 1;
```

### 2. Format des données

#### Structure JSON pour `sites_acces`

```json
[
    {
        "code": "SITE01",
        "nom": "Site Principal",
        "role": "lecteur"
    },
    {
        "code": "SITE02",
        "nom": "Site Secondaire",
        "role": "editeur"
    }
]
```

**Rôles disponibles:**
- `lecteur` - Lecture seule
- `editeur` - Lecture et modification
- `admin` - Tous les droits sur le site

### 3. Migration vers l'API sécurisée

#### Option A: Remplacer l'ancienne API

```bash
# Renommer l'ancienne version
mv pesees.php pesees_old.php

# Utiliser la nouvelle version
mv pesees_secured.php pesees.php
```

#### Option B: Utilisation progressive

Gardez les deux versions et migrez progressivement:

```javascript
// Dans votre code frontend
const API_URL = './pesees_secured.php'; // Nouvelle API sécurisée
// const API_URL = './pesees.php';      // Ancienne API (à supprimer après migration)
```

---

## Utilisation

### Backend (PHP)

#### Vérifier l'accès d'un utilisateur

```php
require_once 'SiteAccessControl.php';

// Récupérer les sites d'un utilisateur
$sites = SiteAccessControl::getUserSites($userId);

// Vérifier l'accès à un site spécifique
if (SiteAccessControl::canAccessSite($userId, 'SITE01')) {
    echo "Accès autorisé";
} else {
    echo "Accès refusé";
}
```

#### Filtrer une requête SQL

```php
// Exemple: Filtrer les pesées
$baseQuery = "SELECT * FROM pesee WHERE 1=1";
$userSites = SiteAccessControl::getUserSites($userId);

$filter = SiteAccessControl::applySiteFilter($baseQuery, $userSites);

$stmt = $pdo->prepare($filter['sql']);
foreach ($filter['params'] as $i => $param) {
    $stmt->bindValue($i + 1, $param);
}
$stmt->execute();
```

#### Assigner des sites à un utilisateur

```php
$sites = [
    ['code' => 'SITE01', 'nom' => 'Site Principal', 'role' => 'lecteur'],
    ['code' => 'SITE02', 'nom' => 'Site Secondaire', 'role' => 'editeur']
];

SiteAccessControl::assignSitesToUser($userId, $sites);
```

### Frontend (JavaScript)

#### Initialisation après le login

```javascript
// Inclure le script
<script src="site-filter-client.js"></script>

// Après un login réussi
fetch('auth.php', {
    method: 'POST',
    body: JSON.stringify({
        action: 'login',
        username: 'user',
        password: 'pass'
    })
})
.then(response => response.json())
.then(data => {
    if (data.success) {
        // Initialiser le module de filtrage
        SiteFilterClient.init(data.data);

        // Les sites sont maintenant disponibles
        console.log('Sites:', SiteFilterClient.getUserSites());
    }
});
```

#### Utilisation avec fetch

```javascript
// Méthode 1: Utiliser le wrapper automatique
SiteFilterClient.fetchWithSiteFilter('pesees.php?action=list')
    .then(response => response.json())
    .then(data => {
        console.log('Pesées filtrées:', data);
    });

// Méthode 2: Ajouter manuellement les paramètres
const url = SiteFilterClient.addSiteParamsToUrl('pesees.php?action=search');
fetch(url)
    .then(response => response.json())
    .then(data => console.log(data));
```

#### Afficher le sélecteur de site

```html
<!-- Dans votre HTML -->
<div id="siteSelectorContainer"></div>

<script>
// Le sélecteur s'affiche automatiquement si l'utilisateur a plusieurs sites
// Vous pouvez aussi l'afficher manuellement:
SiteFilterClient.displaySiteSelector();

// Écouter les changements de site
window.addEventListener('siteChanged', function(event) {
    console.log('Site changé:', event.detail.site);
    // Recharger les données
    loadPesees();
});
</script>
```

---

## API Reference

### Endpoints disponibles

#### 1. Sites API (`sites.php`)

##### GET - Lister tous les sites disponibles

```http
GET /sites.php?action=list
Authorization: Bearer {token}
```

**Réponse:**
```json
{
    "success": true,
    "data": {
        "sites": [
            {"code": "SITE01", "nom": "Site Principal"},
            {"code": "SITE02", "nom": "Site Secondaire"}
        ],
        "count": 2
    }
}
```

##### GET - Sites d'un utilisateur

```http
GET /sites.php?action=user_sites&user_id=1
Authorization: Bearer {token}
```

##### POST - Assigner des sites

```http
POST /sites.php
Authorization: Bearer {token}
Content-Type: application/json

{
    "user_id": 1,
    "sites": [
        {"code": "SITE01", "nom": "Site Principal", "role": "lecteur"}
    ]
}
```

##### DELETE - Retirer tous les sites

```http
DELETE /sites.php
Authorization: Bearer {token}
Content-Type: application/json

{
    "user_id": 1
}
```

#### 2. Pesées API sécurisée (`pesees_secured.php`)

Tous les endpoints existants sont maintenant filtrés automatiquement:

```http
GET /pesees_secured.php?action=list
Authorization: Bearer {token}

# Retourne uniquement les pesées des sites autorisés
```

### Classe SiteAccessControl

#### Méthodes publiques

```php
// Récupérer les sites d'un utilisateur
SiteAccessControl::getUserSites(int $userId): array

// Vérifier l'accès à un site
SiteAccessControl::canAccessSite(int $userId, string $codeSite): bool

// Appliquer un filtre SQL
SiteAccessControl::applySiteFilter(string $query, array $userSites): array

// Assigner des sites
SiteAccessControl::assignSitesToUser(int $userId, array $sites): bool

// Supprimer tous les sites
SiteAccessControl::removeAllSites(int $userId): bool

// Lister tous les sites disponibles
SiteAccessControl::getAllAvailableSites(): array

// Obtenir des statistiques
SiteAccessControl::getSiteStatistics(): array
```

---

## Exemples

### Exemple 1: Workflow complet d'assignation

```php
// 1. Récupérer tous les sites disponibles
$availableSites = SiteAccessControl::getAllAvailableSites();

// 2. Sélectionner les sites à assigner
$sitesToAssign = [
    ['code' => 'SITE01', 'nom' => 'Site Principal', 'role' => 'lecteur'],
    ['code' => 'SITE02', 'nom' => 'Site Secondaire', 'role' => 'lecteur']
];

// 3. Assigner à l'utilisateur
$userId = 5;
SiteAccessControl::assignSitesToUser($userId, $sitesToAssign);

// 4. Vérifier l'assignation
$userSites = SiteAccessControl::getUserSites($userId);
echo "Utilisateur $userId a accès à " . count($userSites) . " sites";
```

### Exemple 2: Application dans une requête de recherche

```php
// Récupérer l'utilisateur depuis le JWT
$payload = JwtManager::requireAuth();
$userId = $payload['user_id'];

// Récupérer ses sites
$userSites = SiteAccessControl::getUserSites($userId);

// Construire la requête de base
$baseQuery = "SELECT * FROM pesee WHERE NomClient LIKE ?";
$params = ['%ACME%'];

// Appliquer le filtre de sites
$filter = SiteAccessControl::applySiteFilter($baseQuery, $userSites);

// Préparer et exécuter
$stmt = $pdo->prepare($filter['sql']);

// Bind des paramètres de recherche
$paramIndex = 1;
foreach ($params as $param) {
    $stmt->bindValue($paramIndex++, $param);
}

// Bind des paramètres de sites
foreach ($filter['params'] as $param) {
    $stmt->bindValue($paramIndex++, $param);
}

$stmt->execute();
$results = $stmt->fetchAll();
```

### Exemple 3: Interface client avec changement de site

```html
<!DOCTYPE html>
<html>
<head>
    <script src="site-filter-client.js"></script>
</head>
<body>
    <div id="siteSelectorContainer"></div>
    <div id="resultsContainer"></div>

    <script>
    // Initialiser après le login
    async function login() {
        const response = await fetch('auth.php', {
            method: 'POST',
            body: JSON.stringify({
                action: 'login',
                username: 'user',
                password: 'pass'
            })
        });

        const data = await response.json();

        if (data.success) {
            // Initialiser le filtrage par site
            SiteFilterClient.init(data.data);

            // Charger les données
            loadData();
        }
    }

    // Charger les données avec filtrage
    async function loadData() {
        const response = await SiteFilterClient.fetchWithSiteFilter(
            'pesees_secured.php?action=list'
        );

        const data = await response.json();

        if (data.success) {
            displayResults(data.data.pesees);
        }
    }

    // Écouter les changements de site
    window.addEventListener('siteChanged', function(event) {
        console.log('Rechargement pour le site:', event.detail.site);
        loadData();
    });

    function displayResults(pesees) {
        const html = pesees.map(p => `
            <div>${p.CodePesee} - ${p.NomClient} - ${p.NomSite}</div>
        `).join('');

        document.getElementById('resultsContainer').innerHTML = html;
    }
    </script>
</body>
</html>
```

---

## Dépannage

### Problème: Les sites ne s'affichent pas

**Solution 1:** Vérifier que la colonne existe
```sql
DESCRIBE user;
-- Chercher 'sites_acces' dans la liste
```

**Solution 2:** Vérifier le format JSON
```sql
SELECT NUt, sites_acces FROM user WHERE NUt = 1;
-- Doit être un JSON valide: [{"code":"...","nom":"..."}]
```

### Problème: Erreur "Token invalide"

**Solution:** Vérifier que le token est bien envoyé
```javascript
// Le token doit être dans le header Authorization
fetch('sites.php', {
    headers: {
        'Authorization': 'Bearer ' + localStorage.getItem('token')
    }
});
```

### Problème: L'utilisateur voit tous les sites

**Vérification:**
```sql
-- Vérifier la configuration de l'utilisateur
SELECT
    NUt,
    NomUt,
    sites_acces,
    CASE
        WHEN sites_acces IS NULL OR sites_acces = '[]' THEN 'Accès illimité'
        ELSE 'Accès restreint'
    END as type_acces
FROM user
WHERE NUt = 1;
```

Si `sites_acces` est NULL ou vide, l'utilisateur a accès à tous les sites (comportement normal).

### Problème: Le filtrage ne fonctionne pas

**Debug dans pesees_secured.php:**
```php
// Ajouter des logs temporaires
error_log("User ID: " . $userPayload['user_id']);
error_log("User Sites: " . json_encode($userSites));
error_log("SQL Query: " . $filter['sql']);
error_log("SQL Params: " . json_encode($filter['params']));
```

### Problème: Erreur de migration SQL

**Solution:** Exécuter étape par étape
```sql
-- 1. Vérifier d'abord
SELECT * FROM user LIMIT 1;

-- 2. Tester l'UPDATE sur un seul utilisateur
UPDATE user
SET sites_acces = '[{"code":"TEST","nom":"Test","role":"lecteur"}]'
WHERE NUt = 1;

-- 3. Vérifier le résultat
SELECT sites_acces FROM user WHERE NUt = 1;
```

---

## Sécurité

### Bonnes pratiques

1. ✅ **Toujours utiliser JWT** pour l'authentification
2. ✅ **Valider les entrées** avant de les stocker en base
3. ✅ **Filtrer côté serveur** (ne jamais se fier uniquement au filtrage client)
4. ✅ **Logger les changements** d'affectation de sites
5. ✅ **Limiter les administrateurs** qui peuvent gérer les sites

### Points de vigilance

- ⚠️ Un utilisateur sans site (`sites_acces = NULL`) a accès à **TOUS** les sites
- ⚠️ Le filtrage client-side est pour l'UX, la sécurité est côté serveur
- ⚠️ Tester les permissions après chaque modification

---

## Support

### Fichiers de log

```bash
# Logs PHP
tail -f pesees-api/logs/error.log

# Logs du navigateur
# Ouvrir la Console (F12) et chercher les erreurs
```

### Tests unitaires

```php
// Tester SiteAccessControl
require_once 'SiteAccessControl.php';

// Test 1: Récupérer les sites
$sites = SiteAccessControl::getUserSites(1);
var_dump($sites);

// Test 2: Vérifier l'accès
$canAccess = SiteAccessControl::canAccessSite(1, 'SITE01');
var_dump($canAccess);

// Test 3: Filtrage SQL
$filter = SiteAccessControl::applySiteFilter(
    "SELECT * FROM pesee WHERE 1=1",
    [['code' => 'SITE01']]
);
var_dump($filter);
```

---

## Changelog

### Version 1.0.0 (2025-10-08)

- ✅ Création du système de compartimentalisation par sites
- ✅ Migration de la structure de base de données
- ✅ API de gestion des sites (sites.php)
- ✅ API sécurisée des pesées (pesees_secured.php)
- ✅ Module JavaScript de filtrage client
- ✅ Interface d'administration (admin_sites.html)
- ✅ Modification de auth.php pour inclure les sites au login
- ✅ Documentation complète

---

## Prochaines étapes

### Fonctionnalités à venir

- [ ] Rôles avancés (lecteur/éditeur/admin par site)
- [ ] Historique des changements d'affectation
- [ ] Notifications lors de l'ajout/retrait de sites
- [ ] Export/Import des configurations de sites
- [ ] Dashboard statistiques par site
- [ ] API GraphQL pour requêtes complexes

---

**📚 Pour plus d'informations, consultez les fichiers sources:**
- `SiteAccessControl.php` - Documentation inline des méthodes
- `site-filter-client.js` - Documentation JSDoc des fonctions
- `migration_sites.sql` - Commentaires détaillés du schéma

**👨‍💻 Développé pour OZIXSOFT - Gestion des Pesées**
