// push-manager.js - Gestionnaire de notifications push pour KOMBAR FERD
// Système complet de notifications push intégré à l'authentification JWT

class PushNotificationManager {
    
    constructor() {
        this.isSupported = 'serviceWorker' in navigator && 'PushManager' in window;
        this.subscription = null;
        this.publicKey = null; // Sera configuré côté serveur
        this.isEnabled = false;
        
        console.log('🔔 PushNotificationManager initialisé');
        console.log('📱 Support notifications:', this.isSupported);
    }
    
    /**
     * Initialiser le système de notifications
     */
    async initialize() {
        if (!this.isSupported) {
            console.warn('⚠️ Notifications push non supportées');
            return false;
        }
        
        try {
            // Vérifier les permissions actuelles
            const permission = await this.checkPermission();
            console.log('🔐 Permission notifications:', permission);
            
            if (permission === 'granted') {
                await this.setupSubscription();
            }
            
            return true;
        } catch (error) {
            console.error('❌ Erreur initialisation notifications:', error);
            return false;
        }
    }
    
    /**
     * Vérifier les permissions de notification
     */
    async checkPermission() {
        if (!('Notification' in window)) {
            return 'unsupported';
        }
        
        return Notification.permission;
    }
    
    /**
     * Demander la permission pour les notifications
     */
    async requestPermission() {
        if (!this.isSupported) {
            throw new Error('Notifications non supportées');
        }
        
        try {
            const permission = await Notification.requestPermission();
            console.log('🔐 Nouvelle permission:', permission);
            
            if (permission === 'granted') {
                await this.setupSubscription();
                this.showPermissionGrantedNotification();
            }
            
            return permission;
        } catch (error) {
            console.error('❌ Erreur demande permission:', error);
            throw error;
        }
    }
    
    /**
     * Configurer l'abonnement push
     */
    async setupSubscription() {
        try {
            const registration = await navigator.serviceWorker.ready;
            console.log('📱 Service Worker prêt pour notifications');
            
            // Vérifier s'il y a déjà un abonnement
            let subscription = await registration.pushManager.getSubscription();
            
            if (!subscription) {
                console.log('🔔 Création nouvel abonnement push...');
                
                try {
                    // Essayer d'abord sans clé VAPID (mode local)
                    subscription = await registration.pushManager.subscribe({
                        userVisibleOnly: true
                    });
                    console.log('✅ Abonnement push créé (mode local sans VAPID)');
                } catch (localError) {
                    console.warn('⚠️ Échec mode local, tentative avec VAPID...');
                    
                    // Fallback avec clé VAPID
                    const publicKey = this.generatePublicKey();
                    subscription = await registration.pushManager.subscribe({
                        userVisibleOnly: true,
                        applicationServerKey: this.urlBase64ToUint8Array(publicKey)
                    });
                    console.log('✅ Abonnement push créé avec VAPID');
                }
            } else {
                console.log('✅ Abonnement push existant trouvé');
            }
            
            this.subscription = subscription;
            this.isEnabled = true;
            
            // Enregistrer l'abonnement côté serveur
            await this.registerSubscription(subscription);
            
            return subscription;
            
        } catch (error) {
            console.error('❌ Erreur setup abonnement:', error);
            throw error;
        }
    }
    
    /**
     * Enregistrer l'abonnement côté serveur
     */
    async registerSubscription(subscription) {
        try {
            console.log('📤 Enregistrement abonnement côté serveur...');
            
            // Obtenir le token JWT actuel
            const token = window.authToken || localStorage.getItem('authToken');
            
            const response = await fetch('./push-subscription', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    action: 'register',
                    subscription: subscription.toJSON(),
                    userAgent: navigator.userAgent,
                    timestamp: new Date().toISOString()
                })
            });
            
            if (response.ok) {
                console.log('✅ Abonnement enregistré côté serveur');
            } else {
                console.warn('⚠️ Erreur enregistrement serveur:', response.statusText);
            }
            
        } catch (error) {
            console.warn('⚠️ Impossible d\'enregistrer côté serveur:', error);
            // Continuer sans serveur (mode local)
        }
    }
    
    /**
     * Désactiver les notifications
     */
    async unsubscribe() {
        try {
            if (this.subscription) {
                const success = await this.subscription.unsubscribe();
                console.log('🔇 Désabonnement:', success ? 'réussi' : 'échoué');
                
                if (success) {
                    this.subscription = null;
                    this.isEnabled = false;
                    
                    // Notifier le serveur
                    await this.unregisterSubscription();
                }
                
                return success;
            }
            
            return true;
        } catch (error) {
            console.error('❌ Erreur désabonnement:', error);
            return false;
        }
    }
    
    /**
     * Supprimer l'abonnement côté serveur
     */
    async unregisterSubscription() {
        try {
            const token = window.authToken || localStorage.getItem('authToken');
            
            await fetch('./push-subscription', {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                }
            });
            
            console.log('✅ Abonnement supprimé côté serveur');
        } catch (error) {
            console.warn('⚠️ Erreur suppression serveur:', error);
        }
    }
    
    /**
     * Envoyer une notification de test
     */
    async sendTestNotification() {
        if (!this.isEnabled) {
            throw new Error('Notifications non activées');
        }
        
        try {
            const token = window.authToken || localStorage.getItem('authToken');
            
            const response = await fetch('./send-notification', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`
                },
                body: JSON.stringify({
                    title: 'KOMBAR FERD - Test',
                    body: 'Notifications push activées avec succès ! 🎉',
                    icon: './icons/icon-192x192.png',
                    badge: './icons/icon-96x96.png',
                    tag: 'test-notification',
                    requireInteraction: false,
                    data: {
                        type: 'test',
                        timestamp: new Date().toISOString(),
                        url: './pesee_webapp.html'
                    }
                })
            });
            
            if (response.ok) {
                console.log('✅ Notification test envoyée');
            } else {
                // Fallback: notification locale
                this.showLocalNotification(
                    'KOMBAR FERD - Test Local',
                    'Notifications locales fonctionnelles ! 📱'
                );
            }
        } catch (error) {
            console.warn('⚠️ Erreur envoi notification serveur, fallback local:', error);
            this.showLocalNotification(
                'KOMBAR FERD - Test Local', 
                'Notifications locales activées ! 📱'
            );
        }
    }
    
    /**
     * Afficher une notification locale
     */
    showLocalNotification(title, body, options = {}) {
        if (Notification.permission === 'granted') {
            const notification = new Notification(title, {
                body: body,
                icon: './icons/icon-192x192.png',
                badge: './icons/icon-96x96.png',
                tag: 'kombar-local',
                requireInteraction: false,
                ...options
            });
            
            notification.onclick = function(event) {
                console.log('🔔 Clic sur notification locale');
                window.focus();
                notification.close();
            };
            
            // Auto-fermer après 5 secondes
            setTimeout(() => {
                notification.close();
            }, 5000);
            
            return notification;
        }
    }
    
    /**
     * Générer une clé publique VAPID (simulation)
     */
    generatePublicKey() {
        // En production, cette clé viendrait du serveur
        return 'BKd5Y8W1vg6hKUBqI7b4fE5ZgZJhx4t8C7u9I2e3D7J4';
    }
    
    /**
     * Convertir clé publique base64 en Uint8Array
     */
    urlBase64ToUint8Array(base64String) {
        const padding = '='.repeat((4 - base64String.length % 4) % 4);
        const base64 = (base64String + padding)
            .replace(/-/g, '+')
            .replace(/_/g, '/');
            
        const rawData = window.atob(base64);
        const outputArray = new Uint8Array(rawData.length);
        
        for (let i = 0; i < rawData.length; ++i) {
            outputArray[i] = rawData.charCodeAt(i);
        }
        
        return outputArray;
    }
    
    /**
     * Afficher notification de permission accordée
     */
    showPermissionGrantedNotification() {
        this.showLocalNotification(
            '🔔 Notifications activées !',
            'Vous recevrez désormais les alertes KOMBAR FERD',
            { tag: 'permission-granted' }
        );
    }
    
    /**
     * Test de notification simple (sans permission)
     */
    testLocalNotification() {
        console.log('🧪 Test notification simple...');
        
        // Essayer d'abord une notification navigateur si possible
        if (Notification.permission === 'granted') {
            this.showLocalNotification(
                '🧪 Test KOMBAR FERD',
                'Notification test réussie ! 🎉',
                { tag: 'test-simple' }
            );
        } else {
            // Sinon, afficher une alerte simulée dans l'interface
            const alertDiv = document.createElement('div');
            alertDiv.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: linear-gradient(135deg, #4facfe, #00f2fe);
                color: white;
                padding: 15px 20px;
                border-radius: 12px;
                box-shadow: 0 10px 25px rgba(0,0,0,0.3);
                z-index: 10000;
                font-weight: 500;
                max-width: 300px;
                animation: slideInRight 0.3s ease-out;
            `;
            alertDiv.innerHTML = `
                <div style="display: flex; align-items: center; gap: 10px;">
                    <span style="font-size: 1.2em;">🧪</span>
                    <div>
                        <div style="font-weight: bold;">Test KOMBAR FERD</div>
                        <div style="font-size: 0.9em; opacity: 0.9;">Notification test réussie !</div>
                    </div>
                    <button onclick="this.parentElement.parentElement.remove()" style="
                        background: rgba(255,255,255,0.2);
                        border: none;
                        color: white;
                        border-radius: 50%;
                        width: 24px;
                        height: 24px;
                        cursor: pointer;
                        font-size: 12px;
                    ">×</button>
                </div>
            `;
            
            document.body.appendChild(alertDiv);
            
            // Auto-supprimer après 4 secondes
            setTimeout(() => {
                if (alertDiv.parentElement) {
                    alertDiv.remove();
                }
            }, 4000);
            
            console.log('✅ Test notification affiché (simulation)');
        }
    }
    
    /**
     * Obtenir le statut des notifications
     */
    getStatus() {
        return {
            isSupported: this.isSupported,
            permission: Notification.permission,
            isEnabled: this.isEnabled,
            hasSubscription: !!this.subscription
        };
    }
    
    /**
     * Interface utilisateur pour gérer les notifications
     */
    createNotificationPanel() {
        const status = this.getStatus();
        console.log('🔍 Status notifications:', status);
        
        const panel = document.createElement('div');
        panel.className = 'notification-panel';
        panel.innerHTML = `
            <div style="
                background: rgba(255, 255, 255, 0.1);
                backdrop-filter: blur(20px);
                border: 1px solid rgba(255, 255, 255, 0.2);
                border-radius: 15px;
                padding: 20px;
                margin: 20px 0;
                color: white;
            ">
                <h3 style="margin: 0 0 15px 0; display: flex; align-items: center;">
                    🔔 Notifications Push
                    <span style="
                        margin-left: 10px;
                        padding: 4px 8px;
                        border-radius: 12px;
                        font-size: 11px;
                        background: ${status.isEnabled ? '#28a745' : '#6c757d'};
                    ">
                        ${status.isEnabled ? 'ACTIVÉES' : 'DÉSACTIVÉES'}
                    </span>
                </h3>
                
                <div style="margin: 10px 0;">
                    <div style="font-size: 14px; margin: 5px 0;">
                        📱 Support: <strong>${status.isSupported ? 'Oui' : 'Non'}</strong>
                    </div>
                    <div style="font-size: 14px; margin: 5px 0;">
                        🔐 Permission: <strong>${this.getPermissionText(status.permission)}</strong>
                    </div>
                    <div style="font-size: 14px; margin: 5px 0;">
                        🔗 Abonnement: <strong>${status.hasSubscription ? 'Actif' : 'Inactif'}</strong>
                    </div>
                </div>
                
                <div style="margin-top: 15px;">
                    ${this.createNotificationButtons(status)}
                </div>
            </div>
        `;
        
        return panel;
    }
    
    /**
     * Obtenir le texte de permission lisible
     */
    getPermissionText(permission) {
        switch (permission) {
            case 'granted': return '✅ Accordée';
            case 'denied': return '❌ Refusée';
            case 'default': return '⏳ En attente';
            default: return '❓ Inconnue';
        }
    }
    
    /**
     * Créer les boutons de gestion
     */
    createNotificationButtons(status) {
        console.log('🔍 Création boutons, status:', status);
        if (!status.isSupported) {
            return '<p style="color: #ffc107;">⚠️ Notifications non supportées par ce navigateur</p>';
        }
        
        let buttons = '';
        
        if (status.permission === 'default' || status.permission === 'denied') {
            buttons += `
                <button onclick="notificationManager.requestPermission()" style="
                    background: linear-gradient(135deg, #28a745, #20c997);
                    border: none;
                    color: white;
                    padding: 10px 20px;
                    border-radius: 25px;
                    cursor: pointer;
                    margin: 5px;
                    font-weight: 500;
                ">
                    🔔 Activer les notifications
                </button>
            `;
        }
        
        // Bouton de test simple toujours disponible
        buttons += `
            <button onclick="notificationManager.testLocalNotification()" style="
                background: linear-gradient(135deg, #ff6b6b, #ee5a52);
                border: none;
                color: white;
                padding: 12px 18px;
                border-radius: 25px;
                cursor: pointer;
                margin: 5px;
                font-weight: 500;
                font-size: 14px;
                width: 100%;
                box-sizing: border-box;
                touch-action: manipulation;
                -webkit-tap-highlight-color: transparent;
            ">
                🧪 Test simple
            </button>
        `;
        
        if (status.isEnabled) {
            buttons += `
                <button onclick="notificationManager.sendTestNotification()" style="
                    background: linear-gradient(135deg, #667eea, #764ba2);
                    border: none;
                    color: white;
                    padding: 10px 20px;
                    border-radius: 25px;
                    cursor: pointer;
                    margin: 5px;
                    font-weight: 500;
                ">
                    🧪 Test notification
                </button>
                <button onclick="notificationManager.unsubscribe().then(() => location.reload())" style="
                    background: linear-gradient(135deg, #dc3545, #c82333);
                    border: none;
                    color: white;
                    padding: 10px 20px;
                    border-radius: 25px;
                    cursor: pointer;
                    margin: 5px;
                    font-weight: 500;
                ">
                    🔇 Désactiver
                </button>
            `;
        }
        
        return buttons;
    }
}

// Instance globale
const notificationManager = new PushNotificationManager();

// Initialisation automatique
document.addEventListener('DOMContentLoaded', async () => {
    await notificationManager.initialize();
    console.log('🔔 Gestionnaire de notifications prêt');
});

// Export pour utilisation dans d'autres scripts
window.notificationManager = notificationManager;